/*
 * @(#)OMClass.java  1.1  2006-06-26
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.oo.objectmodel;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import javax.swing.event.*;
import java.util.*;
import org.jhotdraw.util.ResourceBundleUtil;

/**
 * Simulates the structure and behavior of a class.
 * 
 * @author  Werner Randelshofer
 * @version 1.1 2006-06-26 Support for abstract classes added.
 * <br>1.0 30. Januar 2004  Created.
 */
public class OMClass extends OMObject {

    private String name;
    private boolean isAbstract;
    /**
     * Contains instances of class OMAttribute.
     */
    private ArrayList<OMAttribute> attributes = new ArrayList<OMAttribute>();
    /**
     * Contains instances of class OMOperation.
     */
    private ArrayList<OMOperation> operations = new ArrayList<OMOperation>();

    /** Creates a new instance. */
    public OMClass() {
        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");
        name = labels.getString("model.class.name.text");
    }

    public void setName(String name) {
        if (!this.name.equals(name)) {
            this.name = name;
            fireNameChanged();
        }
    }

    public String getName() {
        return name;
    }

    public void setAbstract(boolean newValue) {
        if (isAbstract != newValue) {
            this.isAbstract = newValue;
            fireAbstractChanged();
        }
    }

    public boolean isAbstract() {
        return isAbstract;
    }

    /**
     * Returns the first generalization of this simulated class.
     * A simulated class should never have more than one generalization.
     */
    public OMGeneralization getGeneralization() {
        Collection list = getRelationships(ObjectModel.GENERALIZATION);
        for (Iterator i = list.iterator(); i.hasNext();) {
            OMGeneralization c = (OMGeneralization) i.next();
            if (c.getStart() == this) {
                return c;
            }
        }
        return null;
    }

    /**
     * Returns all associations defined by this class and all its superclasses.
     */
    public Collection<OMAssociation> getAllAssociations() {
        HashSet<OMAssociation> list = new HashSet(getRelationships(ObjectModel.ASSOCIATION));
        OMClass superclass = getSuperclass();
        if (superclass != null) {
            list.addAll(superclass.getAllAssociations());
        }
        return list;
    }

    public OMClass getSuperclass() {
        OMGeneralization sg = getGeneralization();
        return (sg == null) ? null : (OMClass) sg.getConnected(this);
    }

    public void addRelationship(SimulatedRelationship c) {
        switch (c.getSimulatedConcept()) {
            case ObjectModel.GENERALIZATION:
                super.addRelationship(c);
                if (c.getStart() == this) {
                    for (Iterator i = getAssignableClasses().iterator(); i.hasNext();) {
                        OMClass clazz = (OMClass) i.next();
                        clazz.fireGeneralizationChanged();
                    }
                }
                break;
            default:
                super.addRelationship(c);
                break;
        }
    }

    public void removeRelationship(SimulatedRelationship c) {
        switch (c.getSimulatedConcept()) {
            case ObjectModel.GENERALIZATION:
                if (c.getStart() == this) {
                    Collection ac = getAssignableClasses();
                    super.removeRelationship(c);
                    for (Iterator i = ac.iterator(); i.hasNext();) {
                        OMClass clazz = (OMClass) i.next();
                        clazz.fireGeneralizationChanged();
                    }
                } else {
                    super.removeRelationship(c);
                }
                break;

            default:
                super.removeRelationship(c);
                break;
        }

    }

    /**
     * Determines if the class or interface represented by this Class object is
     * either the same as, or is a superclass or superinterface of, the class or
     * interface represented by the specified Class parameter. It returns true
     * if so; otherwise it returns false.
     */
    public boolean isAssignableFrom(OMClass clazz) {
        OMClass superclass = clazz;

        while (superclass != null && superclass != this) {
            superclass = superclass.getSuperclass();
            if (superclass == clazz) {
                // This code is here to break cyclic subclassing.
                // This should never happen though.
                break;
            }
        }

        return superclass == this;
    }

    /**
     * Returns a collection containing this class and all classes which
     * are assignable from this class.
     * Or in other words, all direct and indirect subclasses of this class.
     */
    public Collection<OMClass> getSubclasses() {
        LinkedList<OMClass> result = new LinkedList<OMClass>();

        Collection list = getRelationships(ObjectModel.GENERALIZATION);
        for (Iterator i = list.iterator(); i.hasNext();) {
            OMGeneralization c = (OMGeneralization) i.next();
            if (c.getEnd() == this) {
                result.add((OMClass) c.getStart());
            }
        }

        return result;
    }

    /**
     * Returns a collection containing this class and all classes which
     * are assignable from this class.
     * Or in other words, all direct and indirect subclasses of this class.
     */
    public Collection<OMClass> getAssignableClasses() {
        // FIXME - Maybe we should add some code which prevents stack overflows
        // on cyclic generalizations.

        LinkedList<OMClass> result = new LinkedList<OMClass>();

        LinkedList<OMClass> resolving;
        LinkedList<OMClass> unresolved = new LinkedList<OMClass>();
        unresolved.add(this);

        while (unresolved.size() > 0) {
            resolving = unresolved;
            unresolved = new LinkedList<OMClass>();
            for (Iterator i = resolving.iterator(); i.hasNext();) {
                OMClass c = (OMClass) i.next();

                result.add(c);
                unresolved.addAll(c.getSubclasses());

            }
        }

        return result;
    }

    public void addAttribute(String name) {
        addAttribute(getAttributeCount(), name);

    }

    public void addAttribute(int index, String name) {
        attributes.add(index, new OMAttribute(this, name));
        fireAttributeAdded(index);
        updateSubclasses();
    }

    public void setAttribute(int index, String name) {
        OMAttribute attr = (OMAttribute) attributes.get(index);
        attr.setName(name);
        fireAttributeChanged(index);
        updateSubclasses();
    }

    /**
     * Returns the attributes declared by this class and all its superclasses.
     */
    public List<OMAttribute> getAllAttributes() {
        LinkedList<OMAttribute> result = new LinkedList<OMAttribute>();

        OMClass clazz = this;
        while (clazz != null) {
            for (int i = clazz.attributes.size() - 1; i >= 0; i--) {
                result.addFirst(clazz.attributes.get(i));
            }
            clazz = clazz.getSuperclass();
        }

        return result;
    }

    public boolean isAttribute(String name) {
        // FIXME - linear search! We should use a map to improve the performance

        for (Iterator i = attributes.iterator(); i.hasNext();) {
            OMAttribute attr = (OMAttribute) i.next();
            if (attr.getAttributeName().equals(name)) {
                return true;
            }
        }

        return false;
    }

    public OMAttribute getAttribute(String name) {
        // FIXME - linear search! We should use a map to improve the performance
        int p = name.indexOf('=');
        if (p != -1) {
            name = name.substring(0, p);
        }
        for (Iterator i = attributes.iterator(); i.hasNext();) {
            OMAttribute attr = (OMAttribute) i.next();
            if (attr.getAttributeName().equals(name)) {
                return attr;
            }
        }

        return null;
    }

    public String getAttributeName(int index) {
        OMAttribute attr = (OMAttribute) attributes.get(index);
        return attr.getAttributeName();
    }

    public OMAttribute getAttribute(int index) {
        return (OMAttribute) attributes.get(index);
    }

    public void removeAttribute(int index) {
        attributes.remove(index);
        fireAttributeRemoved(index);
        updateSubclasses();
    }

    public int getAttributeCount() {
        return attributes.size();
    }

    public void addOperation(String operation) {
        addOperation(getOperationCount(), operation);
    }

    public void addOperation(int index, String name) {
        operations.add(index, new OMOperation(this, name));
        fireOperationAdded(index);
        updateSubclasses();
    }

    public void setOperationName(int index, String name) {
        OMOperation op = (OMOperation) operations.get(index);
        op.setName(name);
        fireOperationChanged(index);
        updateSubclasses();
    }

    public void setOperationMethod(int index, String method) {
        OMOperation op = (OMOperation) operations.get(index);
        op.setMethod(method);
        fireOperationChanged(index);
        updateSubclasses();
    }

    public OMOperation getOperation(int index) {
        return (OMOperation) operations.get(index);
    }

    /**
     * Returns the operations invokable on this class.
     * This includes the operations declared in this class and those inherited
     * by its superclasses, but not overriden operations.
     */
    public List<OMOperation> getInvokableOperations() {
        LinkedList<OMOperation> list = new LinkedList<OMOperation>();

        // Get the operations of our superclass
        if (getSuperclass() != null) {
            list.addAll(getSuperclass().getInvokableOperations());
        }

        // Remove overriden operations from the list of super operations
        for (Iterator i = getOperations().iterator(); i.hasNext();) {
            OMOperation oThis = (OMOperation) i.next();
            for (ListIterator j = list.listIterator(); j.hasNext();) {
                OMOperation oSuper = (OMOperation) j.next();
                if (oSuper.getOperationName().equals(oThis.getOperationName())) {
                    j.remove();
                    break;
                }
            }
        }

        // Add our own operations
        list.addAll(getOperations());

        // Return the result
        return Collections.unmodifiableList(list);
    }

    /**
     * Returns the operations declared by this class.
     */
    public List<OMOperation> getOperations() {
        return Collections.unmodifiableList(operations);
    }

    public void removeOperation(int index) {
        operations.remove(index);
        fireOperationRemoved(index);
        updateSubclasses();
    }

    public int getOperationCount() {
        return operations.size();
    }

    public void addSimulatedClassListener(OMClassListener l) {
        listenerList.add(OMClassListener.class, l);
    }

    public void removeSimulatedClassListener(OMClassListener l) {
        listenerList.remove(OMClassListener.class, l);
    }

    protected void updateSubclasses() {
        for (Iterator i = getRelationships(ObjectModel.GENERALIZATION).iterator(); i.hasNext();) {
            OMGeneralization g = (OMGeneralization) i.next();
            if (g.getSuperclass() == this) {
                g.getSubclass().fireGeneralizationChanged();
                g.getSubclass().updateSubclasses();
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeAdded(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).attributeAdded(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeRemoved(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).attributeRemoved(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeChanged(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).attributeChanged(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireGeneralizationChanged() {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this);
                }
                ((OMClassListener) listeners[i + 1]).generalizationChanged(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireOperationAdded(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).operationAdded(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireOperationRemoved(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).operationRemoved(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireOperationChanged(int index) {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this, index);
                }
                ((OMClassListener) listeners[i + 1]).operationChanged(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireNameChanged() {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this);
                }
                ((OMClassListener) listeners[i + 1]).nameChanged(event);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAbstractChanged() {
        OMClassEvent event = null;

        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == OMClassListener.class) {
                // Lazily create the event:
                if (event == null) {
                    event = new OMClassEvent(this);
                }
                ((OMClassListener) listeners[i + 1]).abstractChanged(event);
            }
        }
    }

    public int getSimulatedConcept() {
        return ObjectModel.CLASS;
    }

    public OMClass clone() {
        OMClass that = (OMClass) super.clone();

        // We have to do a deep copy of the operations
        that.operations = new ArrayList<OMOperation>();
        for (Iterator i = this.operations.iterator(); i.hasNext();) {
            OMOperation op = (OMOperation) i.next();
            that.operations.add((OMOperation) op.clone());
        }

        // We have to do a deep clone of the attributes, because
        // the identity of an attribute is used to determine to which
        // class it belongs.
        that.attributes = new ArrayList<OMAttribute>();
        for (Iterator i = this.attributes.iterator(); i.hasNext();) {
            OMAttribute attr = (OMAttribute) i.next();
            that.attributes.add((OMAttribute) attr.clone());
        }
        return that;
    }

    public void write(DOMOutput out) throws IOException {
        out.addAttribute("name", name);
        out.addAttribute("abstract", isAbstract, false);

        if (attributes.size() > 0) {
            out.openElement("attributes");
            for (Iterator i = attributes.iterator(); i.hasNext();) {
                out.writeObject(i.next());
            }
            out.closeElement();
        }
        if (operations.size() > 0) {
            out.openElement("operations");
            for (Iterator i = operations.iterator(); i.hasNext();) {
                out.writeObject(i.next());
            }
            out.closeElement();
        }
    }

    public void read(DOMInput in) throws IOException {
                ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.oo.Labels");

        name = in.getAttribute("name", labels.getString("model.class.name.text"));
        isAbstract = in.getAttribute("abstract", false);

        if (in.getElementCount("attributes") > 0) {
            in.openElement("attributes");
            for (int i = 0; i < in.getElementCount(); i++) {
                OMAttribute a = (OMAttribute) in.readObject(i);
                a.setDeclaringClass(this);
                attributes.add(a);
            }
            in.closeElement();
        }
        if (in.getElementCount("operations") > 0) {
            in.openElement("operations");
            for (int i = 0; i < in.getElementCount(); i++) {
                OMOperation o = (OMOperation) in.readObject(i);
                o.setDeclaringClass(this);
                operations.add(o);
            }
            in.closeElement();
        }
    }
}
